# Runner

A lightweight CLI tool that polls the longrange.run API for task assignments.

## Overview

Runner is a Cobra-based CLI application that continuously polls the longrange.run API to check for assigned tasks. It provides real-time status updates with clean, readable output.

## Features

- 🔄 Configurable polling interval (default: 15 seconds)
- 🔐 JWT-based authentication via API key
- 📊 Clean, formatted task display with status indicators
- ⏱️ Timestamped polling results
- 🚀 Immediate poll on startup
- 🐳 Docker support

## Installation

### Build from Source

```bash
# From the repository root
go build -o runner ./cmd/runner

# Run the binary
./runner --api-key YOUR_API_KEY
```

### Run with Go

```bash
go run ./cmd/runner/main.go --api-key YOUR_API_KEY
```

### Docker

```bash
# Build the image
docker build -f Dockerfile -t longrange/runner .

# Run the container
docker run longrange/runner --api-key YOUR_API_KEY --model "llama3.2" --poll-interval 30
```



## Usage

### Basic Command

```bash
runner --api-key YOUR_API_KEY
```

### Options

| Flag | Short | Description | Default | Required |
|------|-------|-------------|---------|----------|
| `--api-key` | | JWT API key for authentication | | ✓ Yes |
| `--poll-interval` | | Polling interval in seconds | 15 | No |
| `--model` | | LLM Model | qwen2.5:0.5b | No |

### Examples

**Default polling (15 seconds):**
```bash
runner --api-key sk_1234567890abcdef
```

**Custom polling interval (5 seconds):**
```bash
runner --api-key sk_1234567890abcdef --poll-interval 5
```

**Model Select:**
```bash
runner --api-key sk_1234567890abcdef --model "llama3.2"
```

## Output

### Startup
```
Runner started (API Key: eyJh****f1uM)
Using Ollama model: qwen2.5:0.5b
Polling longrange.run every 30s
```

### Task Display

When tasks are found:
```
[06:16:16] 1 task(s)
  1. [⏳ pending] Some Task
     → Some Response
```

When no tasks are available:
```
[15:04:20] No tasks
```

### Status Indicators

- ⏳ `pending` - Task is queued and waiting to be processed
- ▶ `running` - Task is currently being executed
- ✓ `completed` - Task has finished successfully

### Error Messages

```
✗ Connection failed: dial tcp: connection refused
✗ HTTP 401: {"error":"invalid token"}
✗ Parse error: invalid character '<' looking for beginning of value
```

## API Details

**Endpoint:** `https://longrange.run/api/tasks`

**Method:** `GET`

**Authentication:** Bearer token via `Authorization` header

**Response Format:**
```json
[
  {
    "task_id": "abc123",
    "user_email": "user@example.com",
    "api_key_name": "MyRunner",
    "instructions": "Process data import",
    "status": "pending",
    "created_at": "2024-01-15T10:30:00Z",
    "updated_at": "2024-01-15T10:30:00Z"
  }
]
```

## Configuration

### Environment Variables

While the runner uses command-line flags, you can create a wrapper script with environment variables:

```bash
#!/bin/bash
# run-runner.sh
LONGRANGE_API_KEY="your-api-key-here"
POLL_INTERVAL=15

./runner --api-key "$LONGRANGE_API_KEY" --poll-interval "$POLL_INTERVAL"
```

## Docker Deployment

### Build and Push

```bash
# Build
docker build -f Dockerfile.runner -t longrange/runner:latest .

# Tag for registry
docker tag longrange/runner:latest registry.example.com/longrange/runner:latest

# Push to registry
docker push registry.example.com/longrange/runner:latest
```

### Run as Container

```bash
docker run -d \
  --name longrange-runner \
  --restart unless-stopped \
  longrange/runner:latest \
  --api-key YOUR_API_KEY \
  --poll-interval 30
```

### Docker Compose

```yaml
version: '3.8'

services:
  runner:
    image: longrange/runner:latest
    container_name: longrange-runner
    restart: unless-stopped
    command:
      - --api-key
      - ${LONGRANGE_API_KEY}
      - --poll-interval
      - "30"
```

## Development

### Project Structure

```
cmd/runner/
├── main.go         # Main application code
└── README.md       # This file
```

### Dependencies

- [spf13/cobra](https://github.com/spf13/cobra) - CLI framework
- Go 1.21+ standard library

### Adding Features

The runner uses Cobra for CLI management. Key functions:

- `rootCmd` - Main command definition
- `checkForTasks()` - Polls the API and displays results
- `maskAPIKey()` - Masks sensitive API key in output
- `truncate()` - Truncates long task instructions

## Troubleshooting

**Error: API_KEY is required**
- Ensure you're passing the `--api-key` flag with a valid JWT token

**Error: poll-interval must be at least 1 second**
- The minimum polling interval is 1 second. Adjust your `--poll-interval` value

**✗ HTTP 401: Unauthorized**
- Your API key may be invalid or expired
- Verify your key at longrange.run

**✗ Connection failed**
- Check your internet connection
- Verify the longrange.run API is accessible
- Check for firewall or proxy issues

## License

See repository root for license information.

## Support

For issues and feature requests, please visit the main repository.
